/*
** MPEG Audio stream infomation treating routines
**
** Writen by: Sakae Tatibana <tatibana@extra.hu>
**
** 2000, 3/5 Coding Start
**
*/

#include "binary.h"

#define MP3_INFO_C
#include "mp3_info.h"

int open_mp3_file(char *in, MP3_INFO *out);
int read_mp3_info(FILE *in, MP3_INFO *out);
int close_mp3_file(MP3_INFO *mp3i);
static int get_version(int mpeg_audio_header);
static int get_layer(int mpeg_audio_header);
static int is_stereo(int mpeg_audio_header);
static int get_sample(int mpeg_audio_header);
static int get_bitrate(int mpeg_audio_header);
static int get_padding(int mpeg_audio_header);

int open_mp3_file(char *in, MP3_INFO *out)
{
	out->path = in;
	out->stream = fopen(in, "rb");
	if(out->stream == NULL){
		return 0;
	}

	if(read_mp3_info(out->stream, out)){
        fseek(out->stream, 0, SEEK_SET);
		return 1;
	}else{
		fclose(out->stream);
		return 0;
	}
}

int read_mp3_info(FILE *in, MP3_INFO *out)
{
	int mpeg_audio_header;
	size_t n;
	int pipe;

	out->stream = in;

	if(ftell(in) < 0){
		/* stream is pipe ?  */
		pipe = 1;
	}else{
		fseek(in, 0, SEEK_SET);
		n = ftell(in);
		pipe = 0;
	}

	if(!read_be_int32(in, &mpeg_audio_header)){
		/* can't read MPEG Audio header */
		return 0;
	}

	if((mpeg_audio_header & 0xFFE00000) != 0xFFE00000){
		/* sync header dosen't match */
		return 0;
	}

	if(pipe){
		/* length don't touch */
	}else{
		fseek(in, 0, SEEK_END);
		out->length = ftell(in) - n;
	}

	out->version = get_version(mpeg_audio_header);

	out->layer = get_layer(mpeg_audio_header);
	if(out->layer != MPEG_AUDIO_LAYER_3){
		/* current version can't treat layer 1/2 stream */
		return 0;
	}

	out->bitrate = get_bitrate(mpeg_audio_header);
	out->sample = get_sample(mpeg_audio_header);
	out->stereo = is_stereo(mpeg_audio_header);
    out->padding = get_padding(mpeg_audio_header);

	if(!(out->stereo)){
		out->bitrate /= 2;
	}

	return 1;
}

int close_mp3_file(MP3_INFO *mp3i)
{
	fclose(mp3i->stream);
	return 1;
}

static int get_version(int mpeg_audio_header)
{
	switch(mpeg_audio_header & 0x00180000){
	case 0x00180000:
		return MPEG_VERSION_1;
	case 0x00100000:
		return MPEG_VERSION_2;
	case 0x00000000:
		return MPEG_VERSION_2_5;
	default:
		return MPEG_VERSION_UNKNOWN;
	}
}

static int get_layer(int mpeg_audio_header)
{
	switch(mpeg_audio_header & 0x00060000){
	case 0x00060000:
		return MPEG_AUDIO_LAYER_1;
	case 0x00040000:
		return MPEG_AUDIO_LAYER_2;
	case 0x00020000:
		return MPEG_AUDIO_LAYER_3;
	default:
		return MPEG_AUDIO_LAYER_UNKNOWN;
	}
}

static int is_stereo(int mpeg_audio_header)
{
	switch(mpeg_audio_header & 0x000000C0){
	case 0x000000C0:
		return 0;
	default:
		/* Joint Stereo */
		return 1;
	}
}

static int get_sample(int mpeg_audio_header)
{
	int sample[4][4] = {
		{11025, 12000,  8000, 0},
		{    0,     0,     0, 0},
		{22050, 24000, 16000, 0},
		{44100, 48000, 32000, 0}
	};
	int version;
	int index;

	version = (mpeg_audio_header >> 19) & 3;
	index = (mpeg_audio_header >> 10) & 3;

	return sample[version][index];
}

static int get_bitrate(int mpeg_audio_header)
{
	static const int bitrate[4][16] = {
		{  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160,   0},
		{  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},
		{  0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160,   0},
		{  0,  32,  40,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320,   0}
	};

	int version;
	int index;

	version = (mpeg_audio_header >> 19) & 0x03;
	index = (mpeg_audio_header >> 12) & 0x0F;

	return bitrate[version][index];
}

static int get_padding(int mpeg_audio_header)
{
	if(mpeg_audio_header & 0x00000200){
		return MP3_PADDING_ON;
	}else{
		return MP3_PADDING_OFF;
	}
}
